// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A1__F474

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_96_3_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1100)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xeebbe2a1u, 0x300275c4u, 0, 46 }, // eebbe2a1300275c4 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1100
    { 0xdb286570u, 0x43ba612cu, 0, 63 }, // db28657043ba612c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1100
    { 0x90e10574u, 0xb5af3fc5u, 0, 6 }, // 90e10574b5af3fc5 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1100
    { 0x83f3d435u, 0x0173014eu, 0, 23 }, // 83f3d4350173014e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1100
    { 0xee0f6999u, 0xbe2ea0ebu, 40, 46 }, // ee0f6999be2ea0eb = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1100
    { 0x5ac9b79au, 0x8bdc38fdu, 40, 63 }, // 5ac9b79a8bdc38fd = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1100
    { 0xbf59b533u, 0x31b9aa39u, 40, 80 }, // bf59b53331b9aa39 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1100
    { 0xd572d7fcu, 0x4ad65f05u, 40, 6 }, // d572d7fc4ad65f05 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1100
    { 0xeec9f418u, 0x5a12fb58u, 40, 155 }, // eec9f4185a12fb58 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1100
    { 0x29550cbfu, 0xcf97e243u, 40, 23 }, // 29550cbfcf97e243 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1100
    { 0xd1e87722u, 0x94b2ff83u, 40, 103 }, // d1e8772294b2ff83 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp32@16_96_3_F_T_0___gfx11xx__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1100
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1100_mod0
{{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
 {10, 8, 8,10,10, 8, 8,10, 8, 8},
 {10, 3, 1, 1, 0, 1, 2, 0, 3, 2},
 { 3, 0, 0, 0, 0, 0, 0, 0, 0, 0},
 { 8, 6, 0, 0, 0, 0, 0, 0, 0, 3},
 { 0, 0, 0, 0, 4, 9, 7, 7, 2, 2},
 { 0, 0, 0, 7, 9, 4, 5, 2, 3, 2},
 { 0, 0, 0, 5, 4, 5, 3, 3, 2, 3},
 { 0, 0, 4, 7, 4, 4, 9, 9, 9, 4},
 { 0, 0, 0, 5, 7, 7, 5, 9, 5, 4}}
// End of GPU gfx1100_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A1__F474 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 4;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 4;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = "i32"
// BLOCK_DMODEL = 96
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 96 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

