{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Performs measurements on paths such as determining the length of the path.
-- 
-- Many measuring operations require sampling the path length
-- at intermediate points. Therefore, a @GskPathMeasure@ has
-- a tolerance that determines what precision is required
-- for such approximations.
-- 
-- A @GskPathMeasure@ struct is a reference counted struct
-- and should be treated as opaque.
-- 
-- /Since: 4.14/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Structs.PathMeasure
    ( 

-- * Exported types
    PathMeasure(..)                         ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [ref]("GI.Gsk.Structs.PathMeasure#g:method:ref"), [unref]("GI.Gsk.Structs.PathMeasure#g:method:unref").
-- 
-- ==== Getters
-- [getLength]("GI.Gsk.Structs.PathMeasure#g:method:getLength"), [getPath]("GI.Gsk.Structs.PathMeasure#g:method:getPath"), [getTolerance]("GI.Gsk.Structs.PathMeasure#g:method:getTolerance").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolvePathMeasureMethod                ,
#endif

-- ** getLength #method:getLength#

#if defined(ENABLE_OVERLOADING)
    PathMeasureGetLengthMethodInfo          ,
#endif
    pathMeasureGetLength                    ,


-- ** getPath #method:getPath#

#if defined(ENABLE_OVERLOADING)
    PathMeasureGetPathMethodInfo            ,
#endif
    pathMeasureGetPath                      ,


-- ** getTolerance #method:getTolerance#

#if defined(ENABLE_OVERLOADING)
    PathMeasureGetToleranceMethodInfo       ,
#endif
    pathMeasureGetTolerance                 ,


-- ** new #method:new#

    pathMeasureNew                          ,


-- ** newWithTolerance #method:newWithTolerance#

    pathMeasureNewWithTolerance             ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    PathMeasureRefMethodInfo                ,
#endif
    pathMeasureRef                          ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    PathMeasureUnrefMethodInfo              ,
#endif
    pathMeasureUnref                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Gsk.Callbacks as Gsk.Callbacks
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums
import {-# SOURCE #-} qualified GI.Gsk.Flags as Gsk.Flags
import {-# SOURCE #-} qualified GI.Gsk.Structs.Path as Gsk.Path
import {-# SOURCE #-} qualified GI.Gsk.Structs.Stroke as Gsk.Stroke

#else
import {-# SOURCE #-} qualified GI.Gsk.Structs.Path as Gsk.Path

#endif

-- | Memory-managed wrapper type.
newtype PathMeasure = PathMeasure (SP.ManagedPtr PathMeasure)
    deriving (Eq)

instance SP.ManagedPtrNewtype PathMeasure where
    toManagedPtr (PathMeasure p) = p

foreign import ccall "gsk_path_measure_get_type" c_gsk_path_measure_get_type :: 
    IO GType

type instance O.ParentTypes PathMeasure = '[]
instance O.HasParentTypes PathMeasure

instance B.Types.TypedObject PathMeasure where
    glibType = c_gsk_path_measure_get_type

instance B.Types.GBoxed PathMeasure

-- | Convert t'PathMeasure' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe PathMeasure) where
    gvalueGType_ = c_gsk_path_measure_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr PathMeasure)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr PathMeasure)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed PathMeasure ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList PathMeasure
type instance O.AttributeList PathMeasure = PathMeasureAttributeList
type PathMeasureAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method PathMeasure::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "path"
--           , argType = TInterface Name { namespace = "Gsk" , name = "Path" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path to measure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "PathMeasure" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_path_measure_new" gsk_path_measure_new :: 
    Ptr Gsk.Path.Path ->                    -- path : TInterface (Name {namespace = "Gsk", name = "Path"})
    IO (Ptr PathMeasure)

-- | Creates a measure object for the given /@path@/ with the
-- default tolerance.
-- 
-- /Since: 4.14/
pathMeasureNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Gsk.Path.Path
    -- ^ /@path@/: the path to measure
    -> m PathMeasure
    -- ^ __Returns:__ a new @GskPathMeasure@ representing /@path@/
pathMeasureNew path = liftIO $ do
    path' <- unsafeManagedPtrGetPtr path
    result <- gsk_path_measure_new path'
    checkUnexpectedReturnNULL "pathMeasureNew" result
    result' <- (wrapBoxed PathMeasure) result
    touchManagedPtr path
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method PathMeasure::new_with_tolerance
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "path"
--           , argType = TInterface Name { namespace = "Gsk" , name = "Path" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path to measure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tolerance"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the tolerance for measuring operations"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "PathMeasure" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_path_measure_new_with_tolerance" gsk_path_measure_new_with_tolerance :: 
    Ptr Gsk.Path.Path ->                    -- path : TInterface (Name {namespace = "Gsk", name = "Path"})
    CFloat ->                               -- tolerance : TBasicType TFloat
    IO (Ptr PathMeasure)

-- | Creates a measure object for the given /@path@/ and /@tolerance@/.
-- 
-- /Since: 4.14/
pathMeasureNewWithTolerance ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Gsk.Path.Path
    -- ^ /@path@/: the path to measure
    -> Float
    -- ^ /@tolerance@/: the tolerance for measuring operations
    -> m PathMeasure
    -- ^ __Returns:__ a new @GskPathMeasure@ representing /@path@/
pathMeasureNewWithTolerance path tolerance = liftIO $ do
    path' <- unsafeManagedPtrGetPtr path
    let tolerance' = realToFrac tolerance
    result <- gsk_path_measure_new_with_tolerance path' tolerance'
    checkUnexpectedReturnNULL "pathMeasureNewWithTolerance" result
    result' <- (wrapBoxed PathMeasure) result
    touchManagedPtr path
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method PathMeasure::get_length
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "PathMeasure" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path measure" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFloat)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_path_measure_get_length" gsk_path_measure_get_length :: 
    Ptr PathMeasure ->                      -- self : TInterface (Name {namespace = "Gsk", name = "PathMeasure"})
    IO CFloat

-- | Gets the length of the path being measured.
-- 
-- The length is cached, so this function does not do any work.
-- 
-- /Since: 4.14/
pathMeasureGetLength ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathMeasure
    -- ^ /@self@/: a path measure
    -> m Float
    -- ^ __Returns:__ the length of the path measured by /@self@/
pathMeasureGetLength self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    result <- gsk_path_measure_get_length self'
    let result' = realToFrac result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data PathMeasureGetLengthMethodInfo
instance (signature ~ (m Float), MonadIO m) => O.OverloadedMethod PathMeasureGetLengthMethodInfo PathMeasure signature where
    overloadedMethod = pathMeasureGetLength

instance O.OverloadedMethodInfo PathMeasureGetLengthMethodInfo PathMeasure where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.PathMeasure.pathMeasureGetLength",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-PathMeasure.html#v:pathMeasureGetLength"
        })


#endif

-- method PathMeasure::get_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "PathMeasure" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path measure" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Path" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_path_measure_get_path" gsk_path_measure_get_path :: 
    Ptr PathMeasure ->                      -- self : TInterface (Name {namespace = "Gsk", name = "PathMeasure"})
    IO (Ptr Gsk.Path.Path)

-- | Returns the path that the measure was created for.
-- 
-- /Since: 4.14/
pathMeasureGetPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathMeasure
    -- ^ /@self@/: a path measure
    -> m Gsk.Path.Path
    -- ^ __Returns:__ the path of /@self@/
pathMeasureGetPath self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    result <- gsk_path_measure_get_path self'
    checkUnexpectedReturnNULL "pathMeasureGetPath" result
    result' <- (newBoxed Gsk.Path.Path) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data PathMeasureGetPathMethodInfo
instance (signature ~ (m Gsk.Path.Path), MonadIO m) => O.OverloadedMethod PathMeasureGetPathMethodInfo PathMeasure signature where
    overloadedMethod = pathMeasureGetPath

instance O.OverloadedMethodInfo PathMeasureGetPathMethodInfo PathMeasure where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.PathMeasure.pathMeasureGetPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-PathMeasure.html#v:pathMeasureGetPath"
        })


#endif

-- XXX Could not generate method PathMeasure::get_point
-- Not implemented: Don't know how to allocate "result" of type TInterface (Name {namespace = "Gsk", name = "PathPoint"})
-- method PathMeasure::get_tolerance
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "PathMeasure" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path measure" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFloat)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_path_measure_get_tolerance" gsk_path_measure_get_tolerance :: 
    Ptr PathMeasure ->                      -- self : TInterface (Name {namespace = "Gsk", name = "PathMeasure"})
    IO CFloat

-- | Returns the tolerance that the measure was created with.
-- 
-- /Since: 4.14/
pathMeasureGetTolerance ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathMeasure
    -- ^ /@self@/: a path measure
    -> m Float
    -- ^ __Returns:__ the tolerance of /@self@/
pathMeasureGetTolerance self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    result <- gsk_path_measure_get_tolerance self'
    let result' = realToFrac result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data PathMeasureGetToleranceMethodInfo
instance (signature ~ (m Float), MonadIO m) => O.OverloadedMethod PathMeasureGetToleranceMethodInfo PathMeasure signature where
    overloadedMethod = pathMeasureGetTolerance

instance O.OverloadedMethodInfo PathMeasureGetToleranceMethodInfo PathMeasure where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.PathMeasure.pathMeasureGetTolerance",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-PathMeasure.html#v:pathMeasureGetTolerance"
        })


#endif

-- method PathMeasure::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "PathMeasure" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path measure" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "PathMeasure" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_path_measure_ref" gsk_path_measure_ref :: 
    Ptr PathMeasure ->                      -- self : TInterface (Name {namespace = "Gsk", name = "PathMeasure"})
    IO (Ptr PathMeasure)

-- | Increases the reference count of a @GskPathMeasure@ by one.
-- 
-- /Since: 4.14/
pathMeasureRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathMeasure
    -- ^ /@self@/: a path measure
    -> m PathMeasure
    -- ^ __Returns:__ the passed in @GskPathMeasure@.
pathMeasureRef self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    result <- gsk_path_measure_ref self'
    checkUnexpectedReturnNULL "pathMeasureRef" result
    result' <- (wrapBoxed PathMeasure) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data PathMeasureRefMethodInfo
instance (signature ~ (m PathMeasure), MonadIO m) => O.OverloadedMethod PathMeasureRefMethodInfo PathMeasure signature where
    overloadedMethod = pathMeasureRef

instance O.OverloadedMethodInfo PathMeasureRefMethodInfo PathMeasure where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.PathMeasure.pathMeasureRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-PathMeasure.html#v:pathMeasureRef"
        })


#endif

-- method PathMeasure::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "PathMeasure" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a path measure" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_path_measure_unref" gsk_path_measure_unref :: 
    Ptr PathMeasure ->                      -- self : TInterface (Name {namespace = "Gsk", name = "PathMeasure"})
    IO ()

-- | Decreases the reference count of a @GskPathMeasure@ by one.
-- 
-- If the resulting reference count is zero, frees the object.
-- 
-- /Since: 4.14/
pathMeasureUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    PathMeasure
    -- ^ /@self@/: a path measure
    -> m ()
pathMeasureUnref self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    gsk_path_measure_unref self'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data PathMeasureUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod PathMeasureUnrefMethodInfo PathMeasure signature where
    overloadedMethod = pathMeasureUnref

instance O.OverloadedMethodInfo PathMeasureUnrefMethodInfo PathMeasure where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.PathMeasure.pathMeasureUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-PathMeasure.html#v:pathMeasureUnref"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolvePathMeasureMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolvePathMeasureMethod "ref" o = PathMeasureRefMethodInfo
    ResolvePathMeasureMethod "unref" o = PathMeasureUnrefMethodInfo
    ResolvePathMeasureMethod "getLength" o = PathMeasureGetLengthMethodInfo
    ResolvePathMeasureMethod "getPath" o = PathMeasureGetPathMethodInfo
    ResolvePathMeasureMethod "getTolerance" o = PathMeasureGetToleranceMethodInfo
    ResolvePathMeasureMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolvePathMeasureMethod t PathMeasure, O.OverloadedMethod info PathMeasure p) => OL.IsLabel t (PathMeasure -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolvePathMeasureMethod t PathMeasure, O.OverloadedMethod info PathMeasure p, R.HasField t PathMeasure p) => R.HasField t PathMeasure p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolvePathMeasureMethod t PathMeasure, O.OverloadedMethodInfo info PathMeasure) => OL.IsLabel t (O.MethodProxy info PathMeasure) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


