// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A4__F316

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_160_3_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1200)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xe13bec8eu, 0x0bf347c3u, 0, 80 }, // e13bec8e0bf347c3 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0x3f9f7fc3u, 0x1b791965u, 0, 23 }, // 3f9f7fc31b791965 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0x907cf549u, 0x0d7f1f88u, 0, 57 }, // 907cf5490d7f1f88 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0xa15f2a1eu, 0x435e1a7fu, 0, 143 }, // a15f2a1e435e1a7f = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0xf037bcfcu, 0xc7bbde9cu, 74, 126 }, // f037bcfcc7bbde9c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x2213138fu, 0x6fb1b363u, 74, 143 }, // 2213138f6fb1b363 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0x650381ffu, 0x8344249au, 120, 103 }, // 650381ff8344249a = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0x3431ae96u, 0x4264e251u, 166, 6 }, // 3431ae964264e251 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__64_64__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0x39f7440eu, 0x02036f16u, 166, 126 }, // 39f7440e02036f16 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__64_64__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x526b7b6bu, 0x9f4acb6cu, 166, 143 }, // 526b7b6b9f4acb6c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_160_3_T_F_0___gfx120x__P__64_64__CO__wave4_warp4_stg1--Arch_gfx1200
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1200_mod0
{{1,1,3,2,2,1,1,1,0,1},
 {3,1,3,3,2,1,0,2,1,0},
 {3,0,2,2,2,1,1,0,1,0},
 {3,1,2,2,2,1,0,0,1,3},
 {3,1,2,2,2,2,0,0,3,3},
 {3,1,2,2,2,2,2,3,3,3},
 {3,0,9,2,2,2,5,4,3,3},
 {3,0,7,2,0,3,2,2,2,2},
 {3,0,7,2,3,3,2,2,2,0},
 {3,1,8,6,3,3,0,9,2,0}}
// End of GPU gfx1200_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A4__F316 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 160
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// vim: set fileencoding=utf-8

