mod model;
mod tag_row;

use crate::color::ColorRGBA;
use crate::gobject_models::{GSidebarSelection, GTag};
use crate::util::{GtkUtil, constants};
use diffus::edit::{Edit, collection};
use gdk4::Display;
use gio::ListStore;
use glib::{Object, SignalHandlerId, subclass::*};
use gtk4::{Box, CompositeTemplate, CssProvider, SingleSelection, Widget, prelude::*, subclass::prelude::*};
use gtk4::{ListItem, STYLE_PROVIDER_PRIORITY_APPLICATION};
pub use model::TagListModel;
use once_cell::sync::Lazy;
use std::cell::RefCell;
use tag_row::TagRow;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate)]
    #[template(file = "data/resources/ui_templates/sidebar/tag_list.blp")]
    pub struct TagList {
        #[template_child]
        pub selection: TemplateChild<SingleSelection>,
        #[template_child]
        pub list_store: TemplateChild<ListStore>,

        pub list_model: RefCell<TagListModel>,
        pub style_provider: RefCell<CssProvider>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for TagList {
        const NAME: &'static str = "TagList";
        type ParentType = Box;
        type Type = super::TagList;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    impl ObjectImpl for TagList {
        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("activated")
                        .action()
                        .param_types([super::TagList::static_type()])
                        .build(),
                    Signal::builder("selection-changed")
                        .param_types([super::TagList::static_type(), GSidebarSelection::static_type()])
                        .build(),
                ]
            });

            SIGNALS.as_ref()
        }
    }

    impl WidgetImpl for TagList {}

    impl BoxImpl for TagList {}

    #[gtk4::template_callbacks]
    impl TagList {
        #[template_callback]
        fn factory_setup(&self, obj: &Object) {
            let Some(list_item) = obj.downcast_ref::<ListItem>() else {
                return;
            };

            let row = TagRow::new();
            list_item.set_child(Some(&row));
        }

        #[template_callback]
        fn factory_bind(&self, obj: &Object) {
            let Some(list_item) = obj.downcast_ref::<ListItem>() else {
                return;
            };
            let Some(tag) = list_item.item().and_downcast::<GTag>() else {
                return;
            };
            let Some(child) = list_item.child().and_downcast::<TagRow>() else {
                return;
            };
            child.bind_model(&tag);
        }

        #[template_callback]
        fn on_listview_activate(&self, _pos: u32) {
            self.obj().activate();
        }

        #[template_callback]
        fn on_selection_changed(&self, _pos: u32, _n_items: u32) {
            let obj = self.obj();
            let pos = self.selection.selected();
            let selection = self
                .selection
                .item(pos)
                .and_downcast::<GTag>()
                .map(|tag| GSidebarSelection::from_tag_list(tag, pos));
            if let Some(selection) = selection {
                obj.emit_by_name::<()>("selection-changed", &[&*obj, &selection])
            }
        }
    }
}

glib::wrapper! {
    pub struct TagList(ObjectSubclass<imp::TagList>)
        @extends Widget, Box;
}

impl Default for TagList {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl TagList {
    pub fn new() -> Self {
        Self::default()
    }

    pub fn update(&self, new_list: TagListModel) {
        let imp = self.imp();

        self.update_style_provider(&new_list);

        let old_list = imp.list_model.replace(new_list);

        let model_guard = imp.list_model.borrow();
        let diff = old_list.generate_diff(&model_guard);
        let mut pos = 0;

        match diff {
            Edit::Copy(_list) => {
                // no difference
            }
            Edit::Change(diff) => {
                let _ = diff
                    .into_iter()
                    .map(|edit| {
                        match edit {
                            collection::Edit::Copy(_tag) => {
                                // nothing changed
                                pos += 1;
                            }
                            collection::Edit::Insert(tag) => {
                                imp.list_store.insert(pos, tag);
                                pos += 1;
                            }
                            collection::Edit::Remove(_tag) => {
                                imp.list_store.remove(pos);
                            }
                            collection::Edit::Change(diff) => {
                                let item = imp.list_store.item(pos).and_downcast::<GTag>();
                                if let Some(tag_gobject) = item {
                                    if let Some(label) = diff.label {
                                        tag_gobject.set_label(label.as_str().to_owned());
                                    }
                                    if let Some(color) = diff.color {
                                        tag_gobject.set_color(Some(color.as_str().to_owned()));
                                    }
                                }
                                pos += 1;
                            }
                        }
                    })
                    .collect::<Vec<_>>();
            }
        };
    }

    pub fn clear_selection(&self) {
        self.imp().selection.set_selected(gtk4::INVALID_LIST_POSITION);
    }

    pub fn set_selection(&self, pos: u32) {
        self.imp().selection.set_selected(pos);
    }

    pub fn next_item(&self) -> Option<GSidebarSelection> {
        let imp = self.imp();
        let selected_pos = imp.selection.selected();

        if selected_pos == gtk4::INVALID_LIST_POSITION {
            None
        } else {
            let next_pos = selected_pos + 1;
            imp.selection.set_selected(next_pos);
            imp.selection
                .item(next_pos)
                .and_downcast::<GTag>()
                .map(|tag| GSidebarSelection::from_tag_list(tag, next_pos))
        }
    }

    pub fn prev_item(&self) -> Option<GSidebarSelection> {
        let imp = self.imp();
        let selected_pos = imp.selection.selected();

        if selected_pos == gtk4::INVALID_LIST_POSITION || selected_pos == 0 {
            None
        } else {
            let prev_pos = selected_pos - 1;
            imp.selection.set_selected(prev_pos);
            imp.selection
                .item(prev_pos)
                .and_downcast::<GTag>()
                .map(|tag| GSidebarSelection::from_tag_list(tag, prev_pos))
        }
    }

    pub fn first_item(&self) -> Option<GSidebarSelection> {
        self.imp()
            .selection
            .item(0)
            .and_downcast::<GTag>()
            .map(|tag| GSidebarSelection::from_tag_list(tag, 0))
    }

    pub fn last_item(&self) -> Option<GSidebarSelection> {
        let imp = self.imp();
        let count = imp.selection.n_items();

        if count > 0 {
            let pos = count - 1;

            imp.selection
                .item(pos)
                .and_downcast::<GTag>()
                .map(|tag| GSidebarSelection::from_tag_list(tag, pos))
        } else {
            None
        }
    }

    pub fn activate(&self) {
        self.emit_by_name::<()>("activated", &[&self.clone()]);
    }

    pub fn connect_activated<F: Fn(&Self) + 'static>(&self, f: F) -> SignalHandlerId {
        self.connect_local("activated", false, move |args| {
            let feed_list = args[1].get::<Self>().expect("The value needs to be of type `TagList`");
            f(&feed_list);
            None
        })
    }

    fn update_style_provider(&self, list_model: &TagListModel) {
        let imp = self.imp();
        gtk4::style_context_remove_provider_for_display(&Display::default().unwrap(), &*imp.style_provider.borrow());

        let mut css_string = String::new();
        for tag_model in list_model.tags() {
            let css_selector = tag_model
                .tag_id()
                .as_ref()
                .as_str()
                .chars()
                .filter(|c| c.is_alphanumeric() && !c.is_whitespace())
                .collect::<String>();
            let css_selector = format!("tag-style-{css_selector}");

            css_string.push_str(&Self::generate_css(&css_selector, tag_model.color().as_deref()));
        }

        let provider = CssProvider::new();
        provider.load_from_string(&css_string);

        gtk4::style_context_add_provider_for_display(
            &Display::default().unwrap(),
            &provider,
            STYLE_PROVIDER_PRIORITY_APPLICATION,
        );
        imp.style_provider.replace(provider);
    }

    fn generate_css(tag_id: &str, color_str: Option<&str>) -> String {
        let tag_color = match ColorRGBA::parse_string(color_str.unwrap_or(constants::TAG_DEFAULT_COLOR)) {
            Ok(color) => color,
            Err(_) => ColorRGBA::parse_string(constants::TAG_DEFAULT_COLOR)
                .expect("Failed to parse default outer RGBA string."),
        };
        let gradient_upper = GtkUtil::adjust_lightness(&tag_color, constants::TAG_GRADIENT_SHIFT, None);
        let gradient_lower = GtkUtil::adjust_lightness(&tag_color, -constants::TAG_GRADIENT_SHIFT, None);

        let gradient_upper_hex = gradient_upper.as_string_no_alpha();
        let gradient_lower_hex = gradient_lower.as_string_no_alpha();

        let background_luminance = tag_color.luminance_normalized();
        let text_shift = if background_luminance > 0.8 {
            background_luminance - 1.3
        } else {
            1.2 - background_luminance
        };
        let mut text_color = tag_color;
        text_color.adjust_lightness_absolute(text_shift).unwrap();
        text_color
            .adjust_saturation_absolute(constants::TAG_TEXT_SATURATION_SHIFT)
            .unwrap();
        let text_color_hex = text_color.as_string_no_alpha();

        let css_str = format!(
            ".{tag_id} {{
                background-image: linear-gradient({gradient_upper_hex}, {gradient_lower_hex});
                color: {text_color_hex};
                border-radius: 12px;
                min-width: 12px;
                padding: 2px 5px 1px;
                opacity: 1.0;
            }}

            "
        );
        css_str
    }
}
