use std::time::Duration;

use chrono::{NaiveTime, Utc};
use gio::{ActionEntry, prelude::*};
use glib::{Variant, VariantType};
use gtk4::prelude::*;
use libadwaita::prelude::*;
use news_flash::error::NewsFlashError;
use news_flash::models::{Article, ArticleFilter, ArticleID, CategoryID, FeedID, Marked, Read, TagID};

use crate::app::App;
use crate::article_list::{ArticleList, MarkUpdate, ReadUpdate};
use crate::article_view::ArticleView;
use crate::content_page::{ArticleListColumn, ArticleListMode, ArticleViewColumn, ContentPage};
use crate::gobject_models::{GArticle, GArticleID, SidebarSelectionType};
use crate::i18n::{i18n, i18n_f};
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::sidebar::{FeedListItemID, SideBar};
use crate::undo_action::UndoAction;

pub struct ArticleActions;

impl ArticleActions {
    pub fn setup() {
        // -------------------------
        // mark all articles as read
        // -------------------------
        let mark_all_read = ActionEntry::builder("mark-all-read")
            .activate(|_win, _action, _parameter| Self::set_all_read())
            .build();

        // -------------------------
        // mark feed as read
        // -------------------------
        let mark_feed_read = ActionEntry::builder("mark-feed-read")
            .parameter_type(Some(&String::static_variant_type()))
            .activate(|_win, _action, parameter| Self::set_feed_read(parameter))
            .build();

        // -------------------------
        // mark category as read
        // -------------------------
        let mark_category_read = ActionEntry::builder("mark-category-read")
            .parameter_type(Some(&String::static_variant_type()))
            .activate(|_win, _action, parameter| Self::set_category_read(parameter))
            .build();

        // -------------------------
        // mark tag as read
        // -------------------------
        let mark_tag_read = ActionEntry::builder("mark-tag-read")
            .parameter_type(Some(&String::static_variant_type()))
            .activate(|_win, _action, parameter| Self::set_tag_read(parameter))
            .build();

        // -------------------------
        // mark today as read
        // -------------------------
        let mark_today_read = ActionEntry::builder("mark-today-read")
            .activate(|_win, _action, _parameter| Self::set_today_read())
            .build();

        // ---------------------------
        // mark sidebar read
        // ---------------------------
        let mark_sidebar_read = ActionEntry::builder("mark-sidebar-read")
            .activate(|_win, _action, _parameter| Self::set_sidebar_read())
            .build();

        // -------------------------
        // set articles unread
        // -------------------------
        let set_articles_unread = ActionEntry::builder("set-articles-unread")
            .parameter_type(VariantType::new("as").as_deref().ok())
            .activate(|_win, _action, parameter| Self::set_articles_unread(parameter))
            .build();

        // -------------------------
        // close article
        // -------------------------
        let close_article = ActionEntry::builder("close-article")
            .activate(|_win, _action, _parameter| Self::close_article())
            .build();

        // -------------------------
        // fullscreen article
        // -------------------------
        let fullscreen_article = ActionEntry::builder("fullscreen-article")
            .activate(|_win, _action, _parameter| Self::fullscreen_article())
            .build();

        // -------------------------
        // set article read
        // -------------------------
        let set_read = ActionEntry::builder("set-article-read")
            .parameter_type(VariantType::new("(si)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::set_article_read(parameter))
            .build();

        // -------------------------
        // set article marked
        // -------------------------
        let set_marked = ActionEntry::builder("set-article-marked")
            .parameter_type(VariantType::new("(si)").as_deref().ok())
            .activate(|_win, _action, parameter| Self::set_article_marked(parameter))
            .build();

        // ---------------------------
        // toggle selected article read
        // ---------------------------
        let toggle_selected_read = ActionEntry::builder("toggle-selected-read")
            .activate(|_win, _action, _parameter| Self::toggle_selected_article_read())
            .build();

        // ---------------------------
        // toggle selected article marked
        // ---------------------------
        let toggle_selected_marked = ActionEntry::builder("toggle-selected-marked")
            .activate(|_win, _action, _parameter| Self::toggle_selected_article_marked())
            .build();

        // ---------------------------
        // copy selected article url
        // ---------------------------
        let copy_article_url = ActionEntry::builder("copy-article-url")
            .activate(|_win, _action, _parameter| Self::copy_article_url())
            .build();

        // ---------------------------
        // scrap article content
        // ---------------------------
        let scrape_content = ActionEntry::builder("scrape-content")
            .activate(|_win, _action, _parameter| Self::scrape_content())
            .build();

        MainWindow::instance().add_action_entries([
            mark_all_read,
            mark_feed_read,
            mark_category_read,
            mark_tag_read,
            mark_today_read,
            mark_sidebar_read,
            set_articles_unread,
            close_article,
            fullscreen_article,
            set_read,
            set_marked,
            toggle_selected_read,
            toggle_selected_marked,
            copy_article_url,
            scrape_content,
        ]);
    }

    fn set_all_read() {
        App::default().set_is_marking_all(true);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                let articles = news_flash.get_articles(ArticleFilter::all_unread())?;
                news_flash.set_all_read(&App::client()).await?;
                Ok(articles)
            },
            Self::mark_read_callback,
        );
    }

    fn set_feed_read(parameter: Option<&Variant>) {
        let Some(feed_id) = parameter.and_then(Variant::str).map(FeedID::new) else {
            return;
        };

        App::default().set_is_marking_all(true);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                let articles = news_flash.get_articles(ArticleFilter::feed_unread(&feed_id))?;
                news_flash.set_feed_read(&[feed_id], &App::client()).await?;
                Ok(articles)
            },
            Self::mark_read_callback,
        );
    }

    fn set_category_read(parameter: Option<&Variant>) {
        let Some(category_id) = parameter.and_then(Variant::str).map(CategoryID::new) else {
            return;
        };

        App::default().set_is_marking_all(true);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                let articles = news_flash.get_articles(ArticleFilter::category_unread(&category_id))?;
                news_flash.set_category_read(&[category_id], &App::client()).await?;
                Ok(articles)
            },
            Self::mark_read_callback,
        );
    }

    fn set_tag_read(parameter: Option<&Variant>) {
        let Some(tag_id) = parameter.and_then(Variant::str).map(TagID::new) else {
            return;
        };

        App::default().set_is_marking_all(true);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                let articles = news_flash.get_articles(ArticleFilter::tag_unread(&tag_id))?;
                news_flash.set_tag_read(&[tag_id], &App::client()).await?;
                Ok(articles)
            },
            Self::mark_read_callback,
        );
    }

    fn set_today_read() {
        App::default().set_is_marking_all(true);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                let now = Utc::now();
                let start = now.with_time(NaiveTime::from_hms_opt(0, 0, 0).unwrap()).unwrap();
                let end = if App::default().settings().article_list().hide_future_articles() {
                    now
                } else {
                    now.with_time(NaiveTime::from_hms_opt(23, 59, 59).unwrap()).unwrap()
                };

                let mut filter = ArticleFilter::all_unread();
                filter.older_than = Some(end);
                filter.newer_than = Some(start);
                let articles = news_flash.get_articles(filter)?;
                let article_ids = articles
                    .iter()
                    .map(|article| article.article_id.clone())
                    .collect::<Vec<_>>();
                news_flash
                    .set_article_read(&article_ids, Read::Read, &App::client())
                    .await?;
                Ok(articles)
            },
            Self::mark_read_callback,
        );
    }

    fn set_filter_read(article_filter: ArticleFilter) {
        App::default().set_is_marking_all(true);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;

                let articles = news_flash.get_articles(article_filter)?;
                let article_ids = articles
                    .iter()
                    .map(|article| article.article_id.clone())
                    .collect::<Vec<_>>();
                news_flash
                    .set_article_read(&article_ids, Read::Read, &App::client())
                    .await?;
                Ok(articles)
            },
            Self::mark_read_callback,
        );
    }

    fn set_sidebar_read() {
        let sidebar_selection = SideBar::instance().selection();
        let search_term = ArticleListColumn::instance().search_term();
        let search_term = if search_term.is_empty() {
            None
        } else {
            Some(search_term)
        };
        let list_mode = ArticleListColumn::instance().mode();

        if search_term.is_none() && list_mode != ArticleListMode::Marked {
            match sidebar_selection.selection_type() {
                SidebarSelectionType::None => {}
                SidebarSelectionType::All => MainWindow::activate_action("mark-all-read", None),
                SidebarSelectionType::Today => MainWindow::activate_action("mark-today-read", None),
                SidebarSelectionType::FeedList => match sidebar_selection.feedlist_id().as_ref() {
                    FeedListItemID::Feed(feed_mapping) => {
                        MainWindow::activate_action("mark-feed-read", Some(&feed_mapping.feed_id.as_str().to_variant()))
                    }
                    FeedListItemID::Category(category_id) => {
                        MainWindow::activate_action("mark-category-read", Some(&category_id.as_str().to_variant()))
                    }
                },
                SidebarSelectionType::TagList => MainWindow::activate_action(
                    "mark-tag-read",
                    Some(&sidebar_selection.tag_id().as_ref().as_str().to_variant()),
                ),
            }
        } else {
            let mut filter = ArticleFilter {
                search_term,
                marked: match list_mode {
                    ArticleListMode::All => None,
                    ArticleListMode::Unread => None,
                    ArticleListMode::Marked => Some(Marked::Marked),
                },
                ..Default::default()
            };
            match sidebar_selection.selection_type() {
                SidebarSelectionType::All | SidebarSelectionType::None => {}
                SidebarSelectionType::Today => {
                    let now = Utc::now();
                    let start = now.with_time(NaiveTime::from_hms_opt(0, 0, 0).unwrap()).unwrap();
                    let end = if App::default().settings().article_list().hide_future_articles() {
                        now
                    } else {
                        now.with_time(NaiveTime::from_hms_opt(23, 59, 59).unwrap()).unwrap()
                    };
                    filter.older_than = Some(end);
                    filter.newer_than = Some(start);
                }
                SidebarSelectionType::FeedList => match sidebar_selection.feedlist_id().as_ref() {
                    FeedListItemID::Feed(feed_mapping) => filter.feeds = Some(vec![feed_mapping.feed_id.clone()]),
                    FeedListItemID::Category(category_id) => filter.categories = Some(vec![category_id.clone()]),
                },
                SidebarSelectionType::TagList => filter.tags = Some(vec![sidebar_selection.tag_id().into()]),
            }
            Self::set_filter_read(filter)
        }
    }

    fn mark_read_callback(res: Result<Vec<Article>, NewsFlashError>) {
        match res {
            Err(error) => {
                let message = i18n("Failed to mark read");
                log::error!("{message}");
                ContentPage::instance().newsflash_error(&message, error);
            }
            Ok(articles) => {
                let undo_action = UndoAction::MarkRead(articles.into_iter().map(|a| a.article_id).collect());
                ContentPage::instance().add_undo_notification(undo_action);
            }
        }

        App::default().set_is_marking_all(false);
        ArticleViewColumn::instance().refresh_article_metadata_from_db();
        if ArticleListColumn::instance().mode() == ArticleListMode::Unread {
            ArticleListColumn::instance().new_list();
        } else {
            ArticleListColumn::instance().update_list();
        }
        ContentPage::instance().update_sidebar();
    }

    fn set_articles_unread(parameter: Option<&Variant>) {
        let Some(vec): Option<Vec<String>> = parameter.and_then(FromVariant::from_variant) else {
            return;
        };
        let article_ids: Vec<ArticleID> = vec.into_iter().map(ArticleID::from_owned).collect();

        App::default().set_is_marking_all(true);

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash
                    .set_article_read(&article_ids, Read::Unread, &App::client())
                    .await
            },
            |res| {
                if let Err(error) = res {
                    let message = i18n("Failed to mark multiple article ids read");
                    log::error!("{message}");
                    ContentPage::instance().newsflash_error(&message, error);
                }

                App::default().set_is_marking_all(false);
                ArticleViewColumn::instance().refresh_article_metadata_from_db();
                ArticleListColumn::instance().new_list();
                ContentPage::instance().update_sidebar();
            },
        );
    }

    fn close_article() {
        ArticleViewColumn::instance().set_article(GArticle::NONE);
        MainWindow::instance().set_fullscreened(false);
    }

    fn fullscreen_article() {
        let is_fullscreened = MainWindow::instance().is_fullscreen();
        MainWindow::instance().set_fullscreened(!is_fullscreened);
        if let Some(article_navigation_page) = ContentPage::instance().inner().content() {
            article_navigation_page.set_can_pop(is_fullscreened);
        }
    }

    fn toggle_selected_article_read() {
        // get selected article from list
        let selected_article = ArticleList::instance().selected_model();

        let update = if let Some(selected_article) = selected_article {
            Some(ReadUpdate {
                article_id: selected_article.article_id(),
                read: selected_article.read().invert(),
            })
        } else {
            // if no selected article in list check article view next
            ArticleViewColumn::instance().article().map(|article| ReadUpdate {
                article_id: article.article_id(),
                read: article.read().invert(),
            })
        };

        if let Some(update) = update {
            Self::set_article_read(Some(&update.to_variant()));
        }
    }

    fn toggle_selected_article_marked() {
        // get selected article from list
        let selected_article = ArticleList::instance().selected_model();

        let update = if let Some(selected_article) = selected_article {
            Some(MarkUpdate {
                article_id: selected_article.article_id(),
                marked: selected_article.marked().invert(),
            })
        } else {
            // if no selected article in list check article view next
            ArticleViewColumn::instance().article().map(|article| MarkUpdate {
                article_id: article.article_id(),
                marked: article.marked().invert(),
            })
        };

        if let Some(update) = update {
            Self::set_article_marked(Some(&update.to_variant()));
        }
    }

    fn set_article_read(parameter: Option<&Variant>) {
        let Some(update) = parameter.and_then(ReadUpdate::from_variant) else {
            return;
        };

        ArticleList::instance().set_article_row_state(&update.article_id, Some(update.read), None);

        let article_id_vec = vec![update.article_id.clone().into()];
        let read_status = update.read.into();

        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash
                    .set_article_read(&article_id_vec, read_status, &App::client())
                    .await
            },
            move |res| {
                if let Err(error) = res {
                    let message = i18n_f("Failed to mark article read: '{}'", &[&update.article_id.to_string()]);
                    log::error!("{message}");
                    ContentPage::instance().newsflash_error(&message, error);
                    ArticleListColumn::instance().update_list();
                }

                ContentPage::instance().update_sidebar();

                if let Some(article) = ArticleViewColumn::instance().article()
                    && article.article_id() == update.article_id
                {
                    article.set_read(update.read);
                }
            },
        );
    }

    fn set_article_marked(parameter: Option<&Variant>) {
        let Some(update) = parameter.and_then(MarkUpdate::from_variant) else {
            return;
        };

        ArticleList::instance().set_article_row_state(&update.article_id, None, Some(update.marked));

        let article_id_vec = vec![update.article_id.clone().into()];
        let mark_status = update.marked.into();

        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash
                    .set_article_marked(&article_id_vec, mark_status, &App::client())
                    .await
            },
            move |res| {
                if let Err(error) = res {
                    let message = i18n_f("Failed to star article: '{}'", &[&update.article_id.to_string()]);
                    log::error!("{message}");
                    ContentPage::instance().newsflash_error(&message, error);
                    ArticleListColumn::instance().update_list();
                }

                ContentPage::instance().update_sidebar();

                if let Some(article) = ArticleViewColumn::instance().article()
                    && article.article_id() == update.article_id
                {
                    article.set_marked(update.marked);
                }
            },
        );
    }

    fn copy_article_url() {
        let Some(article_model) = ArticleView::instance().article() else {
            log::warn!("Copy article url to clipboard: No article Selected.");
            return;
        };

        let Some(url) = article_model.url() else {
            log::warn!("Copy article url to clipboard: No url available.");
            return;
        };

        MainWindow::instance().clipboard().set_text(url.as_str());
        ContentPage::instance().simple_message(&i18n("Article URL copied to clipboard"));
    }

    fn scrape_content() {
        let Some(article) = ArticleView::instance().article() else {
            return;
        };

        // Article already scraped: just swap to scraped content
        if article.has_scraped_content() {
            ArticleView::instance().set_prefer_scraped_content(true);
            ArticleView::instance().redraw_article();
            return;
        }

        App::default().set_is_scraping_content(true);

        let article_id: ArticleID = article.article_id().into();
        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;

                if let Some(news_flash) = news_flash_guad.as_ref() {
                    let client = App::client();
                    let news_flash_future = news_flash.scrap_content_article(&article_id, &client, None);
                    tokio::time::timeout(Duration::from_secs(60), news_flash_future).await
                } else {
                    Ok(Err(NewsFlashError::NotLoggedIn))
                }
            },
            |res| {
                App::default().set_is_scraping_content(false);

                match res {
                    Ok(Ok(article)) => {
                        let scraped_article_id: GArticleID = article.article_id.into();
                        let visible_article_id = ArticleView::instance().article().map(|article| article.article_id());

                        if visible_article_id.map(|id| id == scraped_article_id).unwrap_or(false) {
                            ArticleViewColumn::instance().show_article(scraped_article_id, false);
                        }

                        if article.thumbnail_url.is_some() {
                            ArticleListColumn::instance().update_list();
                        }
                    }
                    Ok(Err(error)) => {
                        log::warn!("Internal scraper failed: {error}");
                        ContentPage::instance().newsflash_error(&i18n("Scraper failed to extract content"), error);
                    }
                    Err(_error) => {
                        log::warn!("Internal scraper elapsed");
                        ContentPage::instance().simple_message(&i18n("Scraper Timeout"));
                    }
                }
            },
        );
    }
}
