use super::video_dropdown_row::VideoDropdownRow;
use crate::enclosure::EnclosurePreview;
use crate::gobject_models::{GArticle, GEnclosure};
use gio::ListStore;
use glib::{Object, Properties, clone, subclass::prelude::*, subclass::*};
use gtk4::{
    Box, CompositeTemplate, CustomSorter, ListItem, Ordering, SingleSelection, Widget, prelude::*, subclass::prelude::*,
};
use libadwaita::{Bin, prelude::*, subclass::prelude::*};
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::EnclosurePreviews)]
    #[template(file = "data/resources/ui_templates/enclosures/previews.blp")]
    pub struct EnclosurePreviews {
        #[template_child]
        pub list: TemplateChild<Box>,
        #[template_child]
        pub list_store: TemplateChild<ListStore>,
        #[template_child]
        pub video_preview: TemplateChild<Bin>,
        #[template_child]
        pub selection: TemplateChild<SingleSelection>,
        #[template_child]
        pub sorter: TemplateChild<CustomSorter>,

        #[property(get, set = Self::set_selection)]
        pub selected: Cell<u32>,

        #[property(get, set = Self::set_is_video_fullscreen)]
        pub is_video_fullscreen: Cell<bool>,

        #[property(get, set)]
        pub have_player: Cell<bool>,

        #[property(get, set = Self::set_single_player_item)]
        pub single_player_item: Cell<bool>,

        #[property(get, set)]
        pub show_dropdown: Cell<bool>,

        #[property(get, set = Self::set_article, nullable)]
        pub article: RefCell<Option<GArticle>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for EnclosurePreviews {
        const NAME: &'static str = "EnclosurePreviews";
        type Type = super::EnclosurePreviews;
        type ParentType = Bin;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for EnclosurePreviews {
        fn constructed(&self) {
            self.sorter.set_sort_func(Self::sort_callback);
        }
    }

    impl WidgetImpl for EnclosurePreviews {}

    impl BinImpl for EnclosurePreviews {}

    #[gtk4::template_callbacks]
    impl EnclosurePreviews {
        #[template_callback]
        pub fn on_factory_setup(&self, obj: &Object) {
            let list_item = obj.downcast_ref::<ListItem>().unwrap();
            let row = VideoDropdownRow::default();
            list_item.set_child(Some(&row));
        }

        #[template_callback]
        fn on_factory_bind(&self, obj: &Object) {
            let list_item = obj.downcast_ref::<ListItem>().unwrap();
            let enclosure = list_item.item().and_downcast::<GEnclosure>().unwrap();
            let child = list_item.child().and_downcast::<VideoDropdownRow>().unwrap();
            child.set_model(&enclosure);
        }

        #[template_callback]
        fn set_selection(&self, pos: u32) {
            if let Some(selection) = self.selection.item(pos).and_downcast::<GEnclosure>() {
                let preview = EnclosurePreview::new(&selection);
                self.video_preview.set_child(Some(&preview));
            }

            self.selected.set(pos);
        }

        fn set_article(&self, article: Option<GArticle>) {
            while let Some(child) = self.list.first_child() {
                self.list.remove(&child);
            }
            self.list_store.remove_all();

            let Some(article) = article else {
                self.article.replace(None);
                self.video_preview.set_child(Widget::NONE);
                return;
            };

            let obj = self.obj();

            let audio_video = article
                .enclosures()
                .as_ref()
                .iter()
                .filter(|enclosure| enclosure.is_video() || enclosure.is_audio())
                .cloned()
                .collect::<Vec<_>>();

            obj.set_have_player(!audio_video.is_empty());
            obj.set_single_player_item(audio_video.len() == 1);

            self.list_store.splice(0, 0, &audio_video);

            for enclosure in article
                .enclosures()
                .as_ref()
                .iter()
                .filter(|enclosure| !enclosure.is_video() && !enclosure.is_audio())
            {
                if enclosure.hide_preview() {
                    continue;
                }

                let preview = EnclosurePreview::new(enclosure);
                preview.connect_remove(clone!(
                    #[weak(rename_to = imp)]
                    self,
                    move |preview| imp.list.remove(preview)
                ));
                self.list.append(&preview);
            }
            self.article.replace(Some(article));
        }

        fn set_is_video_fullscreen(&self, is_video_fullscreen: bool) {
            self.obj()
                .set_show_dropdown(!is_video_fullscreen && !self.single_player_item.get());
            self.is_video_fullscreen.set(is_video_fullscreen);
        }

        fn set_single_player_item(&self, single_player_item: bool) {
            self.obj()
                .set_show_dropdown(!single_player_item && !self.is_video_fullscreen.get());
            self.single_player_item.set(single_player_item);
        }

        fn sort_callback(obj1: &Object, obj2: &Object) -> Ordering {
            let enclosure_1 = obj1.downcast_ref::<GEnclosure>().unwrap();
            let enclosure_2 = obj2.downcast_ref::<GEnclosure>().unwrap();

            if (enclosure_1.is_video() && !enclosure_2.is_video()) || enclosure_1.is_default() {
                Ordering::Smaller
            } else if (!enclosure_1.is_video() && enclosure_2.is_video()) || enclosure_2.is_default() {
                Ordering::Larger
            } else if enclosure_1.height() >= enclosure_2.height() || enclosure_1.filesize() >= enclosure_2.filesize() {
                Ordering::Smaller
            } else {
                Ordering::Larger
            }
        }
    }
}

impl Default for EnclosurePreviews {
    fn default() -> Self {
        Object::new()
    }
}

glib::wrapper! {
    pub struct EnclosurePreviews(ObjectSubclass<imp::EnclosurePreviews>)
        @extends Widget, Bin;
}

impl EnclosurePreviews {}
