use crate::enclosure::EnclosurePreviews;
use crate::gobject_models::{GArticle, GDateTime};
use crate::util::DateUtil;
use glib::{Object, Properties, closure, prelude::*, subclass::prelude::*, subclass::*};
use gtk4::{Box, CompositeTemplate, Widget, prelude::*, subclass::prelude::*};
use std::cell::{Cell, RefCell};

mod imp {
    use crate::gobject_models::GOptionalDateTime;

    use super::*;

    #[derive(Default, Debug, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ArticleHeader)]
    #[template(file = "data/resources/ui_templates/article_view/header.blp")]
    pub struct ArticleHeader {
        #[template_child]
        pub enclosure_previews: TemplateChild<EnclosurePreviews>,

        #[property(get, set, nullable)]
        pub article: RefCell<Option<GArticle>>,

        #[property(get, set, name = "feed-name-classes")]
        pub feed_name_classes: RefCell<Vec<String>>,

        #[property(get, set, nullable)]
        pub hovered_url: RefCell<Option<String>>,

        #[property(get, set)]
        pub hide_title: Cell<bool>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleHeader {
        const NAME: &'static str = "ArticleHeader";
        type ParentType = Box;
        type Type = super::ArticleHeader;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
            klass.set_css_name("articleheader");
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ArticleHeader {
        fn constructed(&self) {
            let obj = self.obj();

            obj.property_expression("article")
                .chain_property::<GArticle>("title")
                .chain_closure::<Vec<String>>(closure!(|_: Option<Object>, title: Option<String>| {
                    if title.is_some() {
                        vec!["body".to_string(), "dim-label".to_string()]
                    } else {
                        vec!["heading".to_string(), "dim-label".to_string()]
                    }
                }))
                .bind(&*obj, "feed-name-classes", Widget::NONE);
        }
    }

    impl WidgetImpl for ArticleHeader {}

    impl BoxImpl for ArticleHeader {}

    #[gtk4::template_callbacks]
    impl ArticleHeader {
        #[template_callback]
        fn format_title_markup(&self, title: Option<String>) -> String {
            let url = self
                .article
                .borrow()
                .as_ref()
                .and_then(GArticle::url)
                .unwrap_or_default();
            let title = title.unwrap_or_default();
            let title = glib::markup_escape_text(&title);
            let url = glib::markup_escape_text(&url);
            format!("<a href=\"{url}\">{title}</a>")
        }

        #[template_callback]
        fn format_date(&self, date: GDateTime) -> String {
            DateUtil::format_date(date.as_ref())
        }

        #[template_callback]
        fn format_time(&self, date: GDateTime) -> String {
            DateUtil::format_time(date.as_ref())
        }

        #[template_callback]
        fn string_is_some(&self, string: Option<String>) -> bool {
            string.is_some()
        }

        #[template_callback]
        fn have_updated_date(&self, updated: GOptionalDateTime) -> bool {
            updated.as_ref().is_some()
        }

        #[template_callback]
        fn format_optional_date(&self, date: GOptionalDateTime) -> Option<String> {
            date.as_ref().map(|t| DateUtil::format_date(&t))
        }

        #[template_callback]
        fn format_optional_time(&self, date: GOptionalDateTime) -> Option<String> {
            date.as_ref().map(|t| DateUtil::format_time(&t))
        }

        #[template_callback]
        fn no_title_spacing(&self, title: Option<String>) -> i32 {
            if title.is_some() { 0 } else { 8 }
        }

        #[template_callback]
        fn on_title_enter(&self) {
            let url = self.article.borrow().as_ref().and_then(GArticle::url);
            self.obj().set_hovered_url(url);
        }

        #[template_callback]
        fn on_title_leave(&self) {
            self.obj().set_hovered_url(None::<String>);
        }
    }
}

glib::wrapper! {
    pub struct ArticleHeader(ObjectSubclass<imp::ArticleHeader>)
        @extends Widget, Box;
}

impl Default for ArticleHeader {
    fn default() -> Self {
        Object::new::<Self>()
    }
}
