# frozen_string_literal: true

module Banzai
  module Filter
    # TaskListFilter annotates task list items with aria-labels, creates preceding <task-button>
    # elements, and adds strikethroughs to the text body of inapplicable items (created with `[~]`).
    #
    # This should be run on the HTML generated by the Markdown filter, which handles the actual
    # parsing, after the SanitizationFilter.
    class TaskListFilter < HTML::Pipeline::Filter
      prepend Concerns::PipelineTimingCheck

      CSS   = 'input.task-list-item-checkbox'
      XPATH = Gitlab::Utils::Nokogiri.css_to_xpath(CSS).freeze

      def call
        doc.xpath(XPATH).each do |node|
          text_content = +''
          yield_next_siblings_until(node, %w[ol ul]) do |el|
            text_content << el.text
          end
          truncated_text_content = text_content.strip.truncate(100, separator: ' ', omission: '…')
          node['aria-label'] = format(_('Check option: %{option}'), option: truncated_text_content)

          node.add_previous_sibling(node.document.create_element('task-button'))

          next unless node.has_attribute?('data-inapplicable')

          # We manually apply a <s> to strikethrough text in inapplicable task items,
          # specifically in tight lists where text within the list items isn't contained in a paragraph.
          # (Those are handled entirely by styles.)
          #
          # To handle tight lists, we wrap every text node after the checkbox in <s>, not descending
          # into <p> or <div> (as they're indicative of non-tight lists) or <ul> or <ol> (as we
          # explicitly want to avoid strikethrough styles on sublists, which may have applicable
          # task items!).

          # This is awkward, but we need to include a text node with a space after the input.
          # Otherwise, the strikethrough will start *immediately* next to the <input>, because
          # the first next sibling of the input is always a text node that starts with a space!
          space = node.add_next_sibling(node.document.create_text_node(' '))

          inapplicable_s = node.document.create_element('s')
          inapplicable_s['class'] = 'inapplicable'

          yield_text_nodes_without_descending_into(space.next_sibling, %w[p div ul ol]) do |el|
            el.wrap(inapplicable_s)
          end
        end

        doc
      end

      # Yields the #next_sibling of start, and then the #next_sibling of that, until either
      # there are no more next siblings or a matching element is encountered.
      #
      # The following #next_sibling is evaluated *before* each element is yielded, so they
      # can safely be reparented or removed without affecting iteration.
      def yield_next_siblings_until(start, els)
        it = start.next_sibling
        while it && els.exclude?(it.name)
          following = it.next_sibling
          yield it
          it = following
        end
      end

      # Starting from start, iteratively yield text nodes contained within its children,
      # and its (repeated) #next_siblings and their children, not descending into any of
      # the elements given by els.
      #
      # The following #next_sibling is evaluated before yielding, as above.
      def yield_text_nodes_without_descending_into(start, els)
        stack = [start]
        while stack.any?
          it = stack.pop

          stack << it.next_sibling if it.next_sibling

          if it.text?
            yield it unless it.content.blank?
          elsif els.exclude?(it.name)
            stack.concat(it.children.reverse)
          end
        end
      end
    end
  end
end
