#ifndef IGCC_SPIRV_EXTENSIONS_SUPPORT_H
#define IGCC_SPIRV_EXTENSIONS_SUPPORT_H

#include <vector>
#include <string>
#include "igfxfmid.h"

namespace IGC {
namespace SPIRVExtensionsSupport {

// Helper function for core family hierarchy checks
inline bool isCoreChildOf(PLATFORM Platform, GFXCORE_FAMILY Core) {
  return Platform.eRenderCoreFamily >= Core;
}

// Helper function for product family hierarchy checks
inline bool isProductChildOf(PLATFORM Platform, PRODUCT_FAMILY Product) {
  return Platform.eProductFamily >= Product;
}

// SPIR-V Extension and Capability structures
struct SPIRVCapability {
  std::string Name;
};

struct SPIRVExtension {
  std::string Name;
  std::string SpecURL;
  std::vector<SPIRVCapability> Capabilities;
  bool IsExperimental;
};

// Forward declarations
inline bool isExtensionSupported(const std::string& ExtensionName, PLATFORM Platform);
inline bool isCapabilitySupported(const std::string& CapabilityName, PLATFORM Platform);

// Static vector of all defined extensions with their capabilities
static const std::vector<SPIRVExtension> AllExtensions = {
  {
    "SPV_EXT_float8",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_float8.html",
    {
      {"Float8EXT"},
      {"Float8CooperativeMatrixEXT"}
    },
    true
  },
  {
    "SPV_EXT_optnone",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_optnone.html",
    {
      {"OptNoneEXT"}
    },
    false
  },
  {
    "SPV_EXT_shader_atomic_float16_add",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_shader_atomic_float16_add.html",
    {
      {"AtomicFloat16AddEXT"}
    },
    false
  },
  {
    "SPV_EXT_shader_atomic_float_add",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_shader_atomic_float_add.html",
    {
      {"AtomicFloat32AddEXT"},
      {"AtomicFloat64AddEXT"}
    },
    false
  },
  {
    "SPV_EXT_shader_atomic_float_min_max",
    "https://github.khronos.org/SPIRV-Registry/extensions/EXT/SPV_EXT_shader_atomic_float_min_max.html",
    {
      {"AtomicFloat16MinMaxEXT"},
      {"AtomicFloat32MinMaxEXT"},
      {"AtomicFloat64MinMaxEXT"}
    },
    false
  },
  {
    "SPV_INTEL_16bit_atomics",
    "https://github.com/intel/llvm/pull/20009",
    {
      {"AtomicInt16CompareExchangeINTEL"},
      {"Int16AtomicsINTEL"},
      {"AtomicBFloat16LoadStoreINTEL"},
      {"AtomicBFloat16AddINTEL"},
      {"AtomicBFloat16MinMaxINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_2d_block_io",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_2d_block_io.asciidoc",
    {
      {"Subgroup2DBlockIOINTEL"},
      {"Subgroup2DBlockTransformINTEL"},
      {"Subgroup2DBlockTransposeINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_bfloat16_arithmetic",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_bfloat16_arithmetic.asciidoc",
    {
      {"BFloat16ArithmeticINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_bfloat16_conversion",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_bfloat16_conversion.html",
    {
      {"BFloat16ConversionINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_bindless_images",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_bindless_images.html",
    {
      {"BindlessImagesINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_cache_controls",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_cache_controls.asciidoc",
    {
      {"CacheControlsINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_debug_module",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_debug_module.asciidoc",
    {
      {"DebugInfoModuleINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_float4",
    "https://github.com/intel/llvm/pull/20467",
    {
      {"Float4TypeINTEL"},
      {"Float4CooperativeMatrixINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_fp_conversions",
    "https://github.com/intel/llvm/pull/20467",
    {
      {"FloatConversionsINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_fp_fast_math_mode",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_fp_fast_math_mode.html",
    {
      {"FPFastMathModeINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_fp_max_error",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_fp_max_error.html",
    {
      {"FPMaxErrorINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_function_pointers",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_function_pointers.asciidoc",
    {
      {"FunctionPointersINTEL"},
      {"IndirectReferencesINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_global_variable_host_access",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_global_variable_host_access.html",
    {
      {"GlobalVariableHostAccessINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_inline_assembly",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_inline_assembly.asciidoc",
    {
      {"AsmINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_int4",
    "https://github.com/KhronosGroup/SPIRV-Registry/blob/main/extensions/INTEL/SPV_INTEL_int4.asciidoc",
    {
      {"Int4TypeINTEL"},
      {"Int4CooperativeMatrixINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_joint_matrix",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_joint_matrix.asciidoc",
    {
      {"PackedCooperativeMatrixINTEL"},
      {"CooperativeMatrixInvocationInstructionsINTEL"},
      {"CooperativeMatrixTF32ComponentTypeINTEL"},
      {"CooperativeMatrixBFloat16ComponentTypeINTEL"},
      {"CooperativeMatrixPrefetchINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_kernel_attributes",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_kernel_attributes.html",
    {
      {"KernelAttributesINTEL"},
      {"FPGAKernelAttributesINTEL"},
      {"FPGAKernelAttributesv2INTEL"}
    },
    true
  },
  {
    "SPV_INTEL_long_composites",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_long_composites.html",
    {
      {"CapabilityLongCompositesINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_memory_access_aliasing",
    "https://github.com/intel/llvm/pull/3426/files",
    {
      {"MemoryAccessAliasingINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_predicated_io",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_predicated_io.asciidoc",
    {
      {"PredicatedIOINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_sigmoid",
    "https://github.com/intel/llvm/blob/sycl/sycl/doc/design/spirv-extensions/SPV_INTEL_sigmoid.asciidoc",
    {
      {"SigmoidINTEL"}
    },
    true
  },
  {
    "SPV_INTEL_split_barrier",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_split_barrier.html",
    {
      {"SplitBarrierINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_subgroup_buffer_prefetch",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_subgroup_buffer_prefetch.html",
    {
      {"SubgroupBufferPrefetchINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_subgroup_matrix_multiply_accumulate",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_subgroup_matrix_multiply_accumulate.html",
    {
      {"SubgroupMatrixMultiplyAccumulateINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_subgroups",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_subgroups.html",
    {
      {"SubgroupShuffleINTEL"},
      {"SubgroupBufferBlockIOINTEL"},
      {"SubgroupImageBlockIOINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_tensor_float32_conversion",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_tensor_float32_conversion.html",
    {
      {"TensorFloat32RoundingINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_unstructured_loop_controls",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_unstructured_loop_controls.html",
    {
      {"UnstructuredLoopControlsINTEL"}
    },
    false
  },
  {
    "SPV_INTEL_variable_length_array",
    "https://github.khronos.org/SPIRV-Registry/extensions/INTEL/SPV_INTEL_variable_length_array.html",
    {
      {"VariableLengthArrayINTEL"},
      {"UntypedVariableLengthArrayINTEL"}
    },
    true
  },
  {
    "SPV_KHR_bfloat16",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_bfloat16.html",
    {
      {"BFloat16TypeKHR"},
      {"BFloat16DotProductKHR"},
      {"BFloat16CooperativeMatrixKHR"}
    },
    true
  },
  {
    "SPV_KHR_bit_instructions",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_bit_instructions.html",
    {
      {"BitInstructions"}
    },
    false
  },
  {
    "SPV_KHR_expect_assume",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_expect_assume.html",
    {
      {"ExpectAssumeKHR"}
    },
    false
  },
  {
    "SPV_KHR_integer_dot_product",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_integer_dot_product.html",
    {
      {"DotProductKHR"},
      {"DotProductInputAllKHR"},
      {"DotProductInput4x8BitKHR"},
      {"DotProductInput4x8BitPackedKHR"}
    },
    false
  },
  {
    "SPV_KHR_linkonce_odr",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_linkonce_odr.html",
    {
      {"Linkage"}
    },
    false
  },
  {
    "SPV_KHR_no_integer_wrap_decoration",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_no_integer_wrap_decoration.html",
    {
    },
    false
  },
  {
    "SPV_KHR_shader_clock",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_shader_clock.html",
    {
      {"ShaderClockKHR"}
    },
    false
  },
  {
    "SPV_KHR_uniform_group_instructions",
    "https://github.khronos.org/SPIRV-Registry/extensions/KHR/SPV_KHR_uniform_group_instructions.html",
    {
      {"GroupUniformArithmeticKHR"}
    },
    true
  }
};

// Individual extension/capability query functions
inline bool isExtensionSupported(const std::string& ExtensionName, PLATFORM Platform) {
  if (ExtensionName == "SPV_EXT_float8") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (ExtensionName == "SPV_EXT_optnone") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_EXT_shader_atomic_float16_add") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (ExtensionName == "SPV_EXT_shader_atomic_float_add") {
    return isCapabilitySupported("AtomicFloat32AddEXT", Platform) || isCapabilitySupported("AtomicFloat64AddEXT", Platform);
  }
  if (ExtensionName == "SPV_EXT_shader_atomic_float_min_max") {
    return isCapabilitySupported("AtomicFloat16MinMaxEXT", Platform) || isCapabilitySupported("AtomicFloat32MinMaxEXT", Platform) || isCapabilitySupported("AtomicFloat64MinMaxEXT", Platform);
  }
  if (ExtensionName == "SPV_INTEL_16bit_atomics") {
    return isCapabilitySupported("AtomicInt16CompareExchangeINTEL", Platform) || isCapabilitySupported("Int16AtomicsINTEL", Platform) || isCapabilitySupported("AtomicBFloat16LoadStoreINTEL", Platform) || isCapabilitySupported("AtomicBFloat16AddINTEL", Platform) || isCapabilitySupported("AtomicBFloat16MinMaxINTEL", Platform);
  }
  if (ExtensionName == "SPV_INTEL_2d_block_io") {
    return isCoreChildOf(Platform, IGFX_XE_HPC_CORE);
  }
  if (ExtensionName == "SPV_INTEL_bfloat16_arithmetic") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (ExtensionName == "SPV_INTEL_bfloat16_conversion") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (ExtensionName == "SPV_INTEL_bindless_images") {
    return (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_PVC)) && (!(Platform.eProductFamily == IGFX_CRI));
  }
  if (ExtensionName == "SPV_INTEL_cache_controls") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_debug_module") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_float4") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (ExtensionName == "SPV_INTEL_fp_conversions") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (ExtensionName == "SPV_INTEL_fp_fast_math_mode") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_fp_max_error") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_function_pointers") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_global_variable_host_access") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_inline_assembly") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_int4") {
    return isCapabilitySupported("Int4TypeINTEL", Platform) || isCapabilitySupported("Int4CooperativeMatrixINTEL", Platform);
  }
  if (ExtensionName == "SPV_INTEL_joint_matrix") {
    return isCapabilitySupported("PackedCooperativeMatrixINTEL", Platform) || isCapabilitySupported("CooperativeMatrixInvocationInstructionsINTEL", Platform) || isCapabilitySupported("CooperativeMatrixTF32ComponentTypeINTEL", Platform) || isCapabilitySupported("CooperativeMatrixBFloat16ComponentTypeINTEL", Platform) || isCapabilitySupported("CooperativeMatrixPrefetchINTEL", Platform);
  }
  if (ExtensionName == "SPV_INTEL_kernel_attributes") {
    return isCapabilitySupported("KernelAttributesINTEL", Platform) || isCapabilitySupported("FPGAKernelAttributesINTEL", Platform) || isCapabilitySupported("FPGAKernelAttributesv2INTEL", Platform);
  }
  if (ExtensionName == "SPV_INTEL_long_composites") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_memory_access_aliasing") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_predicated_io") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_sigmoid") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_split_barrier") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_subgroup_buffer_prefetch") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_subgroup_matrix_multiply_accumulate") {
    return (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE)) && (!(Platform.eProductFamily == IGFX_ARROWLAKE));
  }
  if (ExtensionName == "SPV_INTEL_subgroups") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_tensor_float32_conversion") {
    return isCoreChildOf(Platform, IGFX_XE2_HPG_CORE);
  }
  if (ExtensionName == "SPV_INTEL_unstructured_loop_controls") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_INTEL_variable_length_array") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_KHR_bfloat16") {
    return isCoreChildOf(Platform, IGFX_XE2_HPG_CORE);
  }
  if (ExtensionName == "SPV_KHR_bit_instructions") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_KHR_expect_assume") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_KHR_integer_dot_product") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_KHR_linkonce_odr") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_KHR_no_integer_wrap_decoration") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_KHR_shader_clock") {
    return true /* Supported on all platforms */;
  }
  if (ExtensionName == "SPV_KHR_uniform_group_instructions") {
    return true /* Supported on all platforms */;
  }
  return false;
}

inline bool isCapabilitySupported(const std::string& CapabilityName, PLATFORM Platform) {
  if (CapabilityName == "Float8EXT") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "Float8CooperativeMatrixEXT") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "OptNoneEXT") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "AtomicFloat16AddEXT") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "AtomicFloat32AddEXT") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "AtomicFloat64AddEXT") {
    return isProductChildOf(Platform, IGFX_METEORLAKE);
  }
  if (CapabilityName == "AtomicFloat16MinMaxEXT") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "AtomicFloat32MinMaxEXT") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "AtomicFloat64MinMaxEXT") {
    return isProductChildOf(Platform, IGFX_METEORLAKE);
  }
  if (CapabilityName == "AtomicInt16CompareExchangeINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "Int16AtomicsINTEL") {
    return false /* Not supported */;
  }
  if (CapabilityName == "AtomicBFloat16LoadStoreINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "AtomicBFloat16AddINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "AtomicBFloat16MinMaxINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "Subgroup2DBlockIOINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPC_CORE);
  }
  if (CapabilityName == "Subgroup2DBlockTransformINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPC_CORE);
  }
  if (CapabilityName == "Subgroup2DBlockTransposeINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPC_CORE);
  }
  if (CapabilityName == "BFloat16ArithmeticINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "BFloat16ConversionINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "BindlessImagesINTEL") {
    return (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_PVC)) && (!(Platform.eProductFamily == IGFX_CRI));
  }
  if (CapabilityName == "CacheControlsINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "DebugInfoModuleINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "Float4TypeINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "Float4CooperativeMatrixINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "FloatConversionsINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "FPFastMathModeINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "FPMaxErrorINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "FunctionPointersINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "IndirectReferencesINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "GlobalVariableHostAccessINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "AsmINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "Int4TypeINTEL") {
    return isCoreChildOf(Platform, IGFX_XE3P_CORE);
  }
  if (CapabilityName == "Int4CooperativeMatrixINTEL") {
    return false /* Not supported */;
  }
  if (CapabilityName == "PackedCooperativeMatrixINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPG_CORE);
  }
  if (CapabilityName == "CooperativeMatrixInvocationInstructionsINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPG_CORE);
  }
  if (CapabilityName == "CooperativeMatrixTF32ComponentTypeINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPC_CORE);
  }
  if (CapabilityName == "CooperativeMatrixBFloat16ComponentTypeINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPG_CORE);
  }
  if (CapabilityName == "CooperativeMatrixPrefetchINTEL") {
    return isCoreChildOf(Platform, IGFX_XE_HPC_CORE);
  }
  if (CapabilityName == "KernelAttributesINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "FPGAKernelAttributesINTEL") {
    return false /* Not supported */;
  }
  if (CapabilityName == "FPGAKernelAttributesv2INTEL") {
    return false /* Not supported */;
  }
  if (CapabilityName == "CapabilityLongCompositesINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "MemoryAccessAliasingINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "PredicatedIOINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SigmoidINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SplitBarrierINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SubgroupBufferPrefetchINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SubgroupMatrixMultiplyAccumulateINTEL") {
    return (isCoreChildOf(Platform, IGFX_XE_HPG_CORE)) && (!(Platform.eProductFamily == IGFX_METEORLAKE)) && (!(Platform.eProductFamily == IGFX_ARROWLAKE));
  }
  if (CapabilityName == "SubgroupShuffleINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SubgroupBufferBlockIOINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "SubgroupImageBlockIOINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "TensorFloat32RoundingINTEL") {
    return isCoreChildOf(Platform, IGFX_XE2_HPG_CORE);
  }
  if (CapabilityName == "UnstructuredLoopControlsINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "VariableLengthArrayINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "UntypedVariableLengthArrayINTEL") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "BFloat16TypeKHR") {
    return isCoreChildOf(Platform, IGFX_XE2_HPG_CORE);
  }
  if (CapabilityName == "BFloat16DotProductKHR") {
    return isCoreChildOf(Platform, IGFX_XE2_HPG_CORE);
  }
  if (CapabilityName == "BFloat16CooperativeMatrixKHR") {
    return isCoreChildOf(Platform, IGFX_XE2_HPG_CORE);
  }
  if (CapabilityName == "BitInstructions") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "ExpectAssumeKHR") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "DotProductKHR") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "DotProductInputAllKHR") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "DotProductInput4x8BitKHR") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "DotProductInput4x8BitPackedKHR") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "Linkage") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "ShaderClockKHR") {
    return true /* Supported on all platforms */;
  }
  if (CapabilityName == "GroupUniformArithmeticKHR") {
    return true /* Supported on all platforms */;
  }
  return false;
}

// Get extension info with capabilities for a platform
inline std::vector<SPIRVExtension> getSupportedExtensionInfo(PLATFORM Platform, bool includeExperimental = false) {
  std::vector<SPIRVExtension> SupportedExtensions;
  for (const auto& Ext : AllExtensions) {
    if (!includeExperimental && Ext.IsExperimental) {
      continue;
    }
    if (isExtensionSupported(Ext.Name, Platform)) {
      SPIRVExtension SupportedExt;
      SupportedExt.Name = Ext.Name;
      SupportedExt.SpecURL = Ext.SpecURL;
      SupportedExt.IsExperimental = Ext.IsExperimental;
      for (const auto& Cap : Ext.Capabilities) {
        if (isCapabilitySupported(Cap.Name, Platform)) {
          SupportedExt.Capabilities.push_back(Cap);
        }
      }
      SupportedExtensions.push_back(SupportedExt);
    }
  }
  return SupportedExtensions;
}
} // namespace SPIRVExtensionsSupport
} // namespace IGC

#endif // IGCC_SPIRV_EXTENSIONS_SUPPORT_H
