{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The @PangoLanguage@ structure is used to
-- represent a language.
-- 
-- @PangoLanguage@ pointers can be efficiently
-- copied and compared with each other.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Pango.Structs.Language
    ( 

-- * Exported types
    Language(..)                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [includesScript]("GI.Pango.Structs.Language#g:method:includesScript"), [matches]("GI.Pango.Structs.Language#g:method:matches"), [toString]("GI.Pango.Structs.Language#g:method:toString").
-- 
-- ==== Getters
-- [getSampleString]("GI.Pango.Structs.Language#g:method:getSampleString"), [getScripts]("GI.Pango.Structs.Language#g:method:getScripts").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveLanguageMethod                   ,
#endif

-- ** fromString #method:fromString#

    languageFromString                      ,


-- ** getDefault #method:getDefault#

    languageGetDefault                      ,


-- ** getPreferred #method:getPreferred#

    languageGetPreferred                    ,


-- ** getSampleString #method:getSampleString#

#if defined(ENABLE_OVERLOADING)
    LanguageGetSampleStringMethodInfo       ,
#endif
    languageGetSampleString                 ,


-- ** getScripts #method:getScripts#

#if defined(ENABLE_OVERLOADING)
    LanguageGetScriptsMethodInfo            ,
#endif
    languageGetScripts                      ,


-- ** includesScript #method:includesScript#

#if defined(ENABLE_OVERLOADING)
    LanguageIncludesScriptMethodInfo        ,
#endif
    languageIncludesScript                  ,


-- ** matches #method:matches#

#if defined(ENABLE_OVERLOADING)
    LanguageMatchesMethodInfo               ,
#endif
    languageMatches                         ,


-- ** toString #method:toString#

#if defined(ENABLE_OVERLOADING)
    LanguageToStringMethodInfo              ,
#endif
    languageToString                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.Pango.Enums as Pango.Enums

#else
import {-# SOURCE #-} qualified GI.Pango.Enums as Pango.Enums

#endif

-- | Memory-managed wrapper type.
newtype Language = Language (SP.ManagedPtr Language)
    deriving (Eq)

instance SP.ManagedPtrNewtype Language where
    toManagedPtr (Language p) = p

foreign import ccall "pango_language_get_type" c_pango_language_get_type :: 
    IO GType

type instance O.ParentTypes Language = '[]
instance O.HasParentTypes Language

instance B.Types.TypedObject Language where
    glibType = c_pango_language_get_type

instance B.Types.GBoxed Language

-- | Convert t'Language' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Language) where
    gvalueGType_ = c_pango_language_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr Language)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr Language)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed Language ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Language
type instance O.AttributeList Language = LanguageAttributeList
type LanguageAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method Language::get_sample_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "language"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Language" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoLanguage`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_get_sample_string" pango_language_get_sample_string :: 
    Ptr Language ->                         -- language : TInterface (Name {namespace = "Pango", name = "Language"})
    IO CString

-- | Get a string that is representative of the characters needed to
-- render a particular language.
-- 
-- The sample text may be a pangram, but is not necessarily. It is chosen
-- to be demonstrative of normal text in the language, as well as exposing
-- font feature requirements unique to the language. It is suitable for use
-- as sample text in a font selection dialog.
-- 
-- If /@language@/ is 'P.Nothing', the default language as found by
-- 'GI.Pango.Structs.Language.languageGetDefault' is used.
-- 
-- If Pango does not have a sample string for /@language@/, the classic
-- \"The quick brown fox...\" is returned.  This can be detected by
-- comparing the returned pointer value to that returned for (non-existent)
-- language code \"xx\".  That is, compare to:
-- 
-- 
-- 	
-- >pango_language_get_sample_string (pango_language_from_string ("xx"))
languageGetSampleString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Language)
    -- ^ /@language@/: a @PangoLanguage@
    -> m T.Text
    -- ^ __Returns:__ the sample string
languageGetSampleString language = liftIO $ do
    maybeLanguage <- case language of
        Nothing -> return FP.nullPtr
        Just jLanguage -> do
            jLanguage' <- unsafeManagedPtrGetPtr jLanguage
            return jLanguage'
    result <- pango_language_get_sample_string maybeLanguage
    checkUnexpectedReturnNULL "languageGetSampleString" result
    result' <- cstringToText result
    whenJust language touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data LanguageGetSampleStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod LanguageGetSampleStringMethodInfo Language signature where
    overloadedMethod i = languageGetSampleString (Just i)

instance O.OverloadedMethodInfo LanguageGetSampleStringMethodInfo Language where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.Language.languageGetSampleString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-Language.html#v:languageGetSampleString"
        })


#endif

-- method Language::get_scripts
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "language"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Language" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoLanguage`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "num_scripts"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to return number of scripts"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "num_scripts"
--              , argType = TBasicType TInt
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "location to return number of scripts"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just
--               (TCArray
--                  False
--                  (-1)
--                  1
--                  (TInterface Name { namespace = "Pango" , name = "Script" }))
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_get_scripts" pango_language_get_scripts :: 
    Ptr Language ->                         -- language : TInterface (Name {namespace = "Pango", name = "Language"})
    Ptr Int32 ->                            -- num_scripts : TBasicType TInt
    IO (Ptr CInt)

-- | Determines the scripts used to to write /@language@/.
-- 
-- If nothing is known about the language tag /@language@/,
-- or if /@language@/ is 'P.Nothing', then 'P.Nothing' is returned.
-- The list of scripts returned starts with the script that the
-- language uses most and continues to the one it uses least.
-- 
-- The value /@numScript@/ points at will be set to the number
-- of scripts in the returned array (or zero if 'P.Nothing' is returned).
-- 
-- Most languages use only one script for writing, but there are
-- some that use two (Latin and Cyrillic for example), and a few
-- use three (Japanese for example). Applications should not make
-- any assumptions on the maximum number of scripts returned
-- though, except that it is positive if the return value is not
-- 'P.Nothing', and it is a small number.
-- 
-- The 'GI.Pango.Structs.Language.languageIncludesScript' function uses this
-- function internally.
-- 
-- Note: while the return value is declared as @PangoScript@, the
-- returned values are from the @GUnicodeScript@ enumeration, which
-- may have more values. Callers need to handle unknown values.
-- 
-- /Since: 1.22/
languageGetScripts ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Language)
    -- ^ /@language@/: a @PangoLanguage@
    -> m (Maybe [Pango.Enums.Script])
    -- ^ __Returns:__ 
    --   An array of @PangoScript@ values, with the number of entries in
    --   the array stored in /@numScripts@/, or 'P.Nothing' if Pango does not have
    --   any information about this particular language tag (also the case
    --   if /@language@/ is 'P.Nothing').
languageGetScripts language = liftIO $ do
    maybeLanguage <- case language of
        Nothing -> return FP.nullPtr
        Just jLanguage -> do
            jLanguage' <- unsafeManagedPtrGetPtr jLanguage
            return jLanguage'
    numScripts <- allocMem :: IO (Ptr Int32)
    result <- pango_language_get_scripts maybeLanguage numScripts
    numScripts' <- peek numScripts
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (unpackStorableArrayWithLength numScripts') result'
        let result''' = map (toEnum . fromIntegral) result''
        return result'''
    whenJust language touchManagedPtr
    freeMem numScripts
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data LanguageGetScriptsMethodInfo
instance (signature ~ (m (Maybe [Pango.Enums.Script])), MonadIO m) => O.OverloadedMethod LanguageGetScriptsMethodInfo Language signature where
    overloadedMethod i = languageGetScripts (Just i)

instance O.OverloadedMethodInfo LanguageGetScriptsMethodInfo Language where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.Language.languageGetScripts",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-Language.html#v:languageGetScripts"
        })


#endif

-- method Language::includes_script
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "language"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Language" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoLanguage`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "script"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Script" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoScript`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_includes_script" pango_language_includes_script :: 
    Ptr Language ->                         -- language : TInterface (Name {namespace = "Pango", name = "Language"})
    CInt ->                                 -- script : TInterface (Name {namespace = "Pango", name = "Script"})
    IO CInt

-- | Determines if /@script@/ is one of the scripts used to
-- write /@language@/.
-- 
-- The returned value is conservative; if nothing is known about
-- the language tag /@language@/, 'P.True' will be returned, since, as
-- far as Pango knows, /@script@/ might be used to write /@language@/.
-- 
-- This routine is used in Pango\'s itemization process when
-- determining if a supplied language tag is relevant to
-- a particular section of text. It probably is not useful
-- for applications in most circumstances.
-- 
-- This function uses 'GI.Pango.Structs.Language.languageGetScripts' internally.
-- 
-- /Since: 1.4/
languageIncludesScript ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Language)
    -- ^ /@language@/: a @PangoLanguage@
    -> Pango.Enums.Script
    -- ^ /@script@/: a @PangoScript@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@script@/ is one of the scripts used
    --   to write /@language@/ or if nothing is known about /@language@/
    --   (including the case that /@language@/ is 'P.Nothing'), 'P.False' otherwise.
languageIncludesScript language script = liftIO $ do
    maybeLanguage <- case language of
        Nothing -> return FP.nullPtr
        Just jLanguage -> do
            jLanguage' <- unsafeManagedPtrGetPtr jLanguage
            return jLanguage'
    let script' = (fromIntegral . fromEnum) script
    result <- pango_language_includes_script maybeLanguage script'
    let result' = (/= 0) result
    whenJust language touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data LanguageIncludesScriptMethodInfo
instance (signature ~ (Pango.Enums.Script -> m Bool), MonadIO m) => O.OverloadedMethod LanguageIncludesScriptMethodInfo Language signature where
    overloadedMethod i = languageIncludesScript (Just i)

instance O.OverloadedMethodInfo LanguageIncludesScriptMethodInfo Language where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.Language.languageIncludesScript",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-Language.html#v:languageIncludesScript"
        })


#endif

-- method Language::matches
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "language"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Language" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a language tag (see [func@Pango.Language.from_string]),\n  %NULL is allowed and matches nothing but '*'"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "range_list"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a list of language ranges, separated by ';', ':',\n  ',', or space characters.\n  Each element must either be '*', or a RFC 3066 language range\n  canonicalized as by [func@Pango.Language.from_string]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_matches" pango_language_matches :: 
    Ptr Language ->                         -- language : TInterface (Name {namespace = "Pango", name = "Language"})
    CString ->                              -- range_list : TBasicType TUTF8
    IO CInt

-- | Checks if a language tag matches one of the elements in a list of
-- language ranges.
-- 
-- A language tag is considered to match a range in the list if the
-- range is \'*\', the range is exactly the tag, or the range is a prefix
-- of the tag, and the character after it in the tag is \'-\'.
languageMatches ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Language)
    -- ^ /@language@/: a language tag (see 'GI.Pango.Structs.Language.languageFromString'),
    --   'P.Nothing' is allowed and matches nothing but \'*\'
    -> T.Text
    -- ^ /@rangeList@/: a list of language ranges, separated by \';\', \':\',
    --   \',\', or space characters.
    --   Each element must either be \'*\', or a RFC 3066 language range
    --   canonicalized as by 'GI.Pango.Structs.Language.languageFromString'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if a match was found
languageMatches language rangeList = liftIO $ do
    maybeLanguage <- case language of
        Nothing -> return FP.nullPtr
        Just jLanguage -> do
            jLanguage' <- unsafeManagedPtrGetPtr jLanguage
            return jLanguage'
    rangeList' <- textToCString rangeList
    result <- pango_language_matches maybeLanguage rangeList'
    let result' = (/= 0) result
    whenJust language touchManagedPtr
    freeMem rangeList'
    return result'

#if defined(ENABLE_OVERLOADING)
data LanguageMatchesMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m) => O.OverloadedMethod LanguageMatchesMethodInfo Language signature where
    overloadedMethod i = languageMatches (Just i)

instance O.OverloadedMethodInfo LanguageMatchesMethodInfo Language where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.Language.languageMatches",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-Language.html#v:languageMatches"
        })


#endif

-- method Language::to_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "language"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Language" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a language tag." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_to_string" pango_language_to_string :: 
    Ptr Language ->                         -- language : TInterface (Name {namespace = "Pango", name = "Language"})
    IO CString

-- | Gets the RFC-3066 format string representing the given language tag.
-- 
-- Returns (transfer none): a string representing the language tag
languageToString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Language
    -- ^ /@language@/: a language tag.
    -> m T.Text
languageToString language = liftIO $ do
    language' <- unsafeManagedPtrGetPtr language
    result <- pango_language_to_string language'
    checkUnexpectedReturnNULL "languageToString" result
    result' <- cstringToText result
    touchManagedPtr language
    return result'

#if defined(ENABLE_OVERLOADING)
data LanguageToStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod LanguageToStringMethodInfo Language signature where
    overloadedMethod = languageToString

instance O.OverloadedMethodInfo LanguageToStringMethodInfo Language where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.Language.languageToString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-Language.html#v:languageToString"
        })


#endif

-- method Language::from_string
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "language"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string representing a language tag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Language" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_from_string" pango_language_from_string :: 
    CString ->                              -- language : TBasicType TUTF8
    IO (Ptr Language)

-- | Convert a language tag to a @PangoLanguage@.
-- 
-- The language tag must be in a RFC-3066 format. @PangoLanguage@ pointers
-- can be efficiently copied (copy the pointer) and compared with other
-- language tags (compare the pointer.)
-- 
-- This function first canonicalizes the string by converting it to
-- lowercase, mapping \'_\' to \'-\', and stripping all characters other
-- than letters and \'-\'.
-- 
-- Use 'GI.Pango.Structs.Language.languageGetDefault' if you want to get the
-- @PangoLanguage@ for the current locale of the process.
languageFromString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@language@/: a string representing a language tag
    -> m (Maybe Language)
    -- ^ __Returns:__ a @PangoLanguage@
languageFromString language = liftIO $ do
    maybeLanguage <- case language of
        Nothing -> return FP.nullPtr
        Just jLanguage -> do
            jLanguage' <- textToCString jLanguage
            return jLanguage'
    result <- pango_language_from_string maybeLanguage
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Language) result'
        return result''
    freeMem maybeLanguage
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Language::get_default
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Language" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_get_default" pango_language_get_default :: 
    IO (Ptr Language)

-- | Returns the @PangoLanguage@ for the current locale of the process.
-- 
-- On Unix systems, this is the return value is derived from
-- @setlocale (LC_CTYPE, NULL)@, and the user can
-- affect this through the environment variables LC_ALL, LC_CTYPE or
-- LANG (checked in that order). The locale string typically is in
-- the form lang_COUNTRY, where lang is an ISO-639 language code, and
-- COUNTRY is an ISO-3166 country code. For instance, sv_FI for
-- Swedish as written in Finland or pt_BR for Portuguese as written in
-- Brazil.
-- 
-- On Windows, the C library does not use any such environment
-- variables, and setting them won\'t affect the behavior of functions
-- like @/ctime()/@. The user sets the locale through the Regional Options
-- in the Control Panel. The C library (in the @/setlocale()/@ function)
-- does not use country and language codes, but country and language
-- names spelled out in English.
-- However, this function does check the above environment
-- variables, and does return a Unix-style locale string based on
-- either said environment variables or the thread\'s current locale.
-- 
-- Your application should call @setlocale(LC_ALL, \"\")@ for the user
-- settings to take effect. GTK does this in its initialization
-- functions automatically (by calling @/gtk_set_locale()/@).
-- See the @/setlocale()/@ manpage for more details.
-- 
-- Note that the default language can change over the life of an application.
-- 
-- Also note that this function will not do the right thing if you
-- use per-thread locales with @/uselocale()/@. In that case, you should
-- just call 'GI.Pango.Functions.languageFromString' yourself.
-- 
-- /Since: 1.16/
languageGetDefault ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Language
    -- ^ __Returns:__ the default language as a @PangoLanguage@
languageGetDefault  = liftIO $ do
    result <- pango_language_get_default
    checkUnexpectedReturnNULL "languageGetDefault" result
    result' <- (newBoxed Language) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Language::get_preferred
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TCArray
--                  True
--                  (-1)
--                  (-1)
--                  (TInterface Name { namespace = "Pango" , name = "Language" }))
-- throws : False
-- Skip return : False

foreign import ccall "pango_language_get_preferred" pango_language_get_preferred :: 
    IO (Ptr (Ptr Language))

-- | Returns the list of languages that the user prefers.
-- 
-- The list is specified by the @PANGO_LANGUAGE@ or @LANGUAGE@
-- environment variables, in order of preference. Note that this
-- list does not necessarily include the language returned by
-- 'GI.Pango.Structs.Language.languageGetDefault'.
-- 
-- When choosing language-specific resources, such as the sample
-- text returned by 'GI.Pango.Structs.Language.languageGetSampleString',
-- you should first try the default language, followed by the
-- languages returned by this function.
-- 
-- /Since: 1.48/
languageGetPreferred ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m (Maybe [Language])
    -- ^ __Returns:__ a 'P.Nothing'-terminated array
    --   of @PangoLanguage@*
languageGetPreferred  = liftIO $ do
    result <- pango_language_get_preferred
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- unpackZeroTerminatedPtrArray result'
        result''' <- mapM (newBoxed Language) result''
        return result'''
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveLanguageMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveLanguageMethod "includesScript" o = LanguageIncludesScriptMethodInfo
    ResolveLanguageMethod "matches" o = LanguageMatchesMethodInfo
    ResolveLanguageMethod "toString" o = LanguageToStringMethodInfo
    ResolveLanguageMethod "getSampleString" o = LanguageGetSampleStringMethodInfo
    ResolveLanguageMethod "getScripts" o = LanguageGetScriptsMethodInfo
    ResolveLanguageMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveLanguageMethod t Language, O.OverloadedMethod info Language p) => OL.IsLabel t (Language -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveLanguageMethod t Language, O.OverloadedMethod info Language p, R.HasField t Language p) => R.HasField t Language p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveLanguageMethod t Language, O.OverloadedMethodInfo info Language) => OL.IsLabel t (O.MethodProxy info Language) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


